/*
 *  Copyright (C) 2017 The Contributors
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or (at
 *  your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful, but
 *  WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 *  General Public License for more details.
 *
 *  A copy of the GNU General Public License can be found in the file
 *  "gpl.txt" in the top directory of this repository.
 */

#include <all.h>

#define ver(...) _ver(tim_verbose, 0, __VA_ARGS__)
#define ver2(...) _ver(tim_verbose, 1, __VA_ARGS__)
#define ver3(...) _ver(tim_verbose, 2, __VA_ARGS__)

typedef struct {
	uint32_t irq;
	bool run;
	uint32_t latch;
	uint32_t count;
} state_timer;

static state_timer timers[] = {
		{.irq = 0, .run = false, .latch = 32, .count = 32},
		{.irq = 0, .run = false, .latch = 3200, .count = 3200},
		{.irq = 0, .run = false, .latch = 801, .count = 801}
};

#define REG_CRX 0
#define REG_CR2 1
#define REG_T1H	2
#define REG_T1L 3
#define REG_T2H 4
#define REG_T2L 5
#define REG_T3H 6
#define REG_T3L 7

int32_t tim_verbose = 0;

void tim_init(void)
{
	ver("tim init");
}

void tim_quit(void)
{
	ver("tim quit");
}

bool tim_exec(void)
{
	for (int32_t i = 0; i < ARRAY_COUNT(timers); ++i) {
		if(timers[i].run == true) {
			--timers[i].count;
			if(timers[i].count == 0) {
				timers[i].count = timers[i].latch;
				timers[i].irq = 1;
			}
			//ver2("tim%d %u", i, timers[i].count);
		}
	}
	return timers[0].irq || timers[1].irq || timers[2].irq;
}

uint32_t tim_read(uint32_t off, int32_t sz)
{
	ver2("tim rd %u:%d", off, sz * 8);
	uint32_t rv;
	rv = 0;
	switch(off) {
	case REG_CRX:
		break;
	case REG_CR2:
		rv |= (timers[0].irq << 0);
		rv |= (timers[1].irq << 1);
		rv |= (timers[2].irq << 2);
		ver2("tim plc %u fc %u rtc %u", timers[0].irq, timers[1].irq, timers[2].irq);
		//ver2("tim rv %u", rv);
		break;
	case REG_T1H:
		rv = 0;
		break;
	case REG_T1L:
		rv = 31;
		timers[0].irq = 0;
		break;
	case REG_T2H:
		rv = 12;
		break;
	case REG_T2L:
		rv = 127;
		timers[1].irq = 0;
		break;
	case REG_T3H:
		rv = 3;
		break;
	case REG_T3L:
		rv = 32;
		timers[2].irq = 0;
		break;
	default:
		break;
	}
	return rv;
}

void tim_write(uint32_t off, int32_t sz, uint32_t val)
{
	ver2("tim wr %u:%d 0x%0*x", off, sz * 8, sz * 2, val);
	if( off == 0 && (val & (1 << 7)) ) {
		timers[0].run = true;
		timers[1].run = true;
		timers[2].run = true;
	}
}
