/*
 *  Copyright (C) 2017 The Contributors
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or (at
 *  your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful, but
 *  WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 *  General Public License for more details.
 *
 *  A copy of the GNU General Public License can be found in the file
 *  "gpl.txt" in the top directory of this repository.
 */

#include <all.h>

#define ver(...) _ver(cpu_verbose, 0, __VA_ARGS__)
#define ver2(...) _ver(cpu_verbose, 1, __VA_ARGS__)
#define ver3(...) _ver(cpu_verbose, 2, __VA_ARGS__)

int32_t cpu_verbose = 0;

#define CPU_FREQ 7000000
#define PER_SEC 100000

#define APP_START 0x10000

#define RAM_START 0x0
#define RAM_SIZE 0x100000

#define ROM_START 0x100000
#define ROM_SIZE 0x10000

typedef void (*hw_init_t)(void);
typedef void (*hw_quit_t)(void);
typedef void (*hw_exec_t)(void);
typedef uint32_t (*hw_read_t)(uint32_t off, int32_t sz);
typedef void (*hw_write_t)(uint32_t off, int32_t sz, uint32_t val);

typedef struct {
	uint32_t addr_beg;
	uint32_t addr_end;
	hw_init_t init;
	hw_quit_t quit;
	hw_exec_t exec;
	hw_read_t read;
	hw_write_t write;
} hw_t;

static bool reset = true;

static uint8_t ram_data[RAM_SIZE];
static uint8_t rom_data[ROM_SIZE];

static uint32_t ram_ro_beg = 0x1234;
static uint32_t ram_ro_end = 0x1234;
static uint32_t ram_rw_beg = 0x1234;
static uint32_t ram_rw_end = 0x1234;

static uint32_t rom_ro_beg;
static uint32_t rom_ro_end;
static uint32_t rom_rw_beg;
static uint32_t rom_rw_end;

static hw_t hw_map[] = {
	{ 0x180000, 0x200000, fpu_init, fpu_quit, fpu_exec, fpu_read, fpu_write },
	{ 0x200000, 0x280000, vid_init, vid_quit, vid_exec, vid_read, vid_write },
	{ 0x3a0001, 0x3a4001, tim_init, tim_quit, tim_exec, tim_read, tim_write },
	{ 0x3a4001, 0x3a8001, lcd_init, lcd_quit, lcd_exec, lcd_read, lcd_write },
	{ 0x3a8001, 0x3ac001, ser_init, ser_quit, ser_exec, ser_read, ser_write },
	{ 0x3ac001, 0x3b0001, mid_init, mid_quit, mid_exec, mid_read, mid_write },
	{ 0x3b0001, 0x3b4001, fdd_init, fdd_quit, fdd_exec, fdd_read, fdd_write },
	{ 0x3b4001, 0x3b8001, snd_init, snd_quit, snd_exec, snd_read, snd_write },
	{ 0x3b8001, 0x3bc001, led_init, led_quit, led_exec, led_read, led_write },
	{ 0x3bc001, 0x3c0001, kbd_init, kbd_quit, kbd_exec, kbd_read, kbd_write }
};

static hw_t *hw_by_addr(uint32_t addr)
{
	for (int32_t i = 0; i < ARRAY_COUNT(hw_map); ++i) {
		if (addr >= hw_map[i].addr_beg && addr < hw_map[i].addr_end) {
			return hw_map + i;
		}
	}

	return NULL;
}

static void hw_init(void)
{
	inf("initializing hardware");

	for (int32_t i = 0; i < ARRAY_COUNT(hw_map); ++i) {
		hw_map[i].init();
	}
}

static void hw_exec(void)
{
	for (int32_t i = 0; i < ARRAY_COUNT(hw_map); ++i) {
		hw_map[i].exec();
	}
}

static uint32_t hw_off(hw_t *hw, uint32_t addr)
{
	if ((hw->addr_beg & 0x1) == 0) {
		return addr - hw->addr_beg;
	}

	return (addr - hw->addr_beg) / 2;
}

static void bios_init(const char *bios)
{
	inf("loading BIOS file %s", bios);

	SDL_RWops *ops = SDL_RWFromFile(bios, "rb");

	if (ops == NULL) {
		fail("error while opening BIOS file %s", bios);
	}

	if (SDL_ReadBE16(ops) != 0x601b) {
		fail("invalid BIOS file %s", bios);
	}

	uint32_t text_len = SDL_ReadBE32(ops);
	uint32_t data_len = SDL_ReadBE32(ops);
	uint32_t bss_len = SDL_ReadBE32(ops);

	SDL_ReadBE32(ops);
	SDL_ReadBE32(ops);

	uint32_t text_loc = SDL_ReadBE32(ops);

	SDL_ReadBE16(ops);

	uint32_t data_loc = SDL_ReadBE32(ops);
	uint32_t bss_loc = SDL_ReadBE32(ops);

	inf("BIOS text 0x%x:0x%x data 0x%x:0x%x bss 0x%x:0x%x",
			text_loc, text_len, data_loc, data_len, bss_loc, bss_len);

	size_t load_len = (size_t)SDL_RWsize(ops) - 36;

	if (text_loc != ROM_START || text_loc + text_len != data_loc ||
			load_len != text_len + data_len || load_len > ROM_SIZE) {
		fail("invalid BIOS file %s", bios);
	}

	size_t loaded = 0;

	while (loaded < load_len) {
		size_t n_rd = SDL_RWread(ops, rom_data + loaded, 1, load_len - loaded);

		if (n_rd == 0) {
			fail("error while reading BIOS file %s", bios);
		}

		loaded += n_rd;
	}

	SDL_RWclose(ops);

	rom_ro_beg = text_loc;
	rom_ro_end = text_loc + text_len + data_len;
	rom_rw_beg = bss_loc;
	rom_rw_end = bss_loc + bss_len;

	ver("rom_ro_beg 0x%08x rom_ro_end 0x%08x", rom_ro_beg, rom_ro_end);
	ver("rom_rw_beg 0x%08x rom_rw_end 0x%08x", rom_rw_beg, rom_rw_end);
}

uint32_t m68k_read_disassembler_8(uint32_t addr)
{
	return m68k_read_memory_8(addr);
}

uint32_t m68k_read_disassembler_16(uint32_t addr)
{
	return m68k_read_memory_16(addr);
}

uint32_t m68k_read_disassembler_32(uint32_t addr)
{
	return m68k_read_memory_32(addr);
}

uint32_t m68k_read_memory_8(uint32_t addr)
{
	ver3("mem rd 0x%08x:8", addr);

	if (addr >= ram_ro_beg && addr <= ram_ro_end - 1) {
		return ram_data[addr - RAM_START];
	}

	if (addr >= ram_rw_beg && addr <= ram_rw_end - 1) {
		return ram_data[addr - RAM_START];
	}

	if (addr >= rom_ro_beg && addr <= rom_ro_end - 1) {
		return rom_data[addr - ROM_START];
	}

	if (addr >= rom_rw_beg && addr <= rom_rw_end - 1) {
		// ROM has its BSS section in RAM.
		return ram_data[addr - RAM_START];
	}

	hw_t *hw = hw_by_addr(addr);

	if (hw != NULL) {
		return hw->read(hw_off(hw, addr), 1);
	}

	if (addr <= APP_START - 1) {
		return ram_data[addr];
	}

	fail("invalid read 0x%08x:8", addr);
}

uint32_t m68k_read_memory_16(uint32_t addr)
{
	ver3("mem rd 0x%08x:16", addr);

	if (addr >= ram_ro_beg && addr <= ram_ro_end - 2) {
		return
				((uint32_t)ram_data[addr - RAM_START + 0] << 8) |
				((uint32_t)ram_data[addr - RAM_START + 1] << 0);
	}

	if (addr >= ram_rw_beg && addr <= ram_rw_end - 2) {
		return
				((uint32_t)ram_data[addr - RAM_START + 0] << 8) |
				((uint32_t)ram_data[addr - RAM_START + 1] << 0);
	}

	if (addr >= rom_ro_beg && addr <= rom_ro_end - 2) {
		return
				((uint32_t)rom_data[addr - ROM_START + 0] << 8) |
				((uint32_t)rom_data[addr - ROM_START + 1] << 0);
	}

	if (addr >= rom_rw_beg && addr <= rom_rw_end - 2) {
		// ROM has its BSS section in RAM.
		return
				((uint32_t)ram_data[addr - RAM_START + 0] << 8) |
				((uint32_t)ram_data[addr - RAM_START + 1] << 0);
	}

	hw_t *hw = hw_by_addr(addr);

	if (hw != NULL) {
		return hw->read(hw_off(hw, addr), 2);
	}

	if (addr <= APP_START - 2) {
		return
				((uint32_t)ram_data[addr - 0] << 8) |
				((uint32_t)ram_data[addr - 1] << 0);
	}

	fail("invalid read 0x%08x:16", addr);
}

uint32_t m68k_read_memory_32(uint32_t addr)
{
	ver3("mem rd 0x%08x:32", addr);

	if (reset) {
		if (addr == 0) {
			addr += ROM_START;
		}

		else if (addr == 4) {
			addr += ROM_START;
			reset = false;
		}
		else {
			fail("invalid reset sequence");
		}
	}

	if (addr >= ram_ro_beg && addr <= ram_ro_end - 4) {
		return
				((uint32_t)ram_data[addr - RAM_START + 0] << 24) |
				((uint32_t)ram_data[addr - RAM_START + 1] << 16) |
				((uint32_t)ram_data[addr - RAM_START + 2] <<  8) |
				((uint32_t)ram_data[addr - RAM_START + 3] <<  0);
	}

	if (addr >= ram_rw_beg && addr <= ram_rw_end - 4) {
		return
				((uint32_t)ram_data[addr - RAM_START + 0] << 24) |
				((uint32_t)ram_data[addr - RAM_START + 1] << 16) |
				((uint32_t)ram_data[addr - RAM_START + 2] <<  8) |
				((uint32_t)ram_data[addr - RAM_START + 3] <<  0);
	}

	if (addr >= rom_ro_beg && addr <= rom_ro_end - 4) {
		return
				((uint32_t)rom_data[addr - ROM_START + 0] << 24) |
				((uint32_t)rom_data[addr - ROM_START + 1] << 16) |
				((uint32_t)rom_data[addr - ROM_START + 2] <<  8) |
				((uint32_t)rom_data[addr - ROM_START + 3] <<  0);
	}

	if (addr >= rom_rw_beg && addr <= rom_rw_end - 4) {
		// ROM has its BSS section in RAM.
		return
				((uint32_t)ram_data[addr - RAM_START + 0] << 24) |
				((uint32_t)ram_data[addr - RAM_START + 1] << 16) |
				((uint32_t)ram_data[addr - RAM_START + 2] <<  8) |
				((uint32_t)ram_data[addr - RAM_START + 3] <<  0);
	}

	hw_t *hw = hw_by_addr(addr);

	if (hw != NULL) {
		return hw->read(hw_off(hw, addr), 4);
	}

	if (addr <= APP_START - 4) {
		return
				((uint32_t)ram_data[addr + 0] << 24) |
				((uint32_t)ram_data[addr + 1] << 16) |
				((uint32_t)ram_data[addr + 2] <<  8) |
				((uint32_t)ram_data[addr + 3] <<  0);
	}

	fail("invalid read 0x%08x:32", addr);
}

void m68k_write_memory_8(uint32_t addr, uint32_t val)
{
	ver3("mem wr 0x%08x:8 0x%02x", addr, val);

	if (addr >= ram_rw_beg && addr <= ram_rw_end - 1) {
		ram_data[addr - RAM_START] = (uint8_t)val;
		return;
	}

	if (addr >= rom_rw_beg && addr <= rom_rw_end - 1) {
		// ROM has its BSS section in RAM.
		ram_data[addr - RAM_START] = (uint8_t)val;
		return;
	}

	hw_t *hw = hw_by_addr(addr);

	if (hw != NULL) {
		hw->write(hw_off(hw, addr), 1, val);
		return;
	}

	if (addr <= APP_START - 1) {
		ram_data[addr] = (uint8_t)val;
		return;
	}

	fail("invalid write 0x%08x:8 0x%02x", addr, val);
}

void m68k_write_memory_16(uint32_t addr, uint32_t val)
{
	ver3("mem wr 0x%08x:16 0x%04x", addr, val);

	if (addr >= ram_rw_beg && addr <= ram_rw_end - 2) {
		ram_data[addr - RAM_START + 0] = (uint8_t)(val >> 8);
		ram_data[addr - RAM_START + 1] = (uint8_t)(val >> 0);
		return;
	}

	if (addr >= rom_rw_beg && addr <= rom_rw_end - 2) {
		// ROM has its BSS section in RAM.
		ram_data[addr - RAM_START + 0] = (uint8_t)(val >> 8);
		ram_data[addr - RAM_START + 1] = (uint8_t)(val >> 0);
		return;
	}

	hw_t *hw = hw_by_addr(addr);

	if (hw != NULL) {
		hw->write(hw_off(hw, addr), 2, val);
		return;
	}

	if (addr <= APP_START - 2) {
		ram_data[addr + 0] = (uint8_t)(val >> 8);
		ram_data[addr + 1] = (uint8_t)(val >> 0);
		return;
	}

	fail("invalid write 0x%08x:16 0x%04x", addr, val);
}

void m68k_write_memory_32(uint32_t addr, uint32_t val)
{
	ver3("mem wr 0x%08x:32 0x%08x", addr, val);

	if (addr >= ram_rw_beg && addr <= ram_rw_end - 4) {
		ram_data[addr - RAM_START + 0] = (uint8_t)(val >> 24);
		ram_data[addr - RAM_START + 1] = (uint8_t)(val >> 16);
		ram_data[addr - RAM_START + 2] = (uint8_t)(val >>  8);
		ram_data[addr - RAM_START + 3] = (uint8_t)(val >>  0);
		return;
	}

	if (addr >= rom_rw_beg && addr <= rom_rw_end - 4) {
		// ROM has its BSS section in RAM.
		ram_data[addr - RAM_START + 0] = (uint8_t)(val >> 24);
		ram_data[addr - RAM_START + 1] = (uint8_t)(val >> 16);
		ram_data[addr - RAM_START + 2] = (uint8_t)(val >>  8);
		ram_data[addr - RAM_START + 3] = (uint8_t)(val >>  0);
		return;
	}

	hw_t *hw = hw_by_addr(addr);

	if (hw != NULL) {
		hw->write(hw_off(hw, addr), 4, val);
		return;
	}

	if (addr <= APP_START - 4) {
		ram_data[addr + 0] = (uint8_t)(val >> 24);
		ram_data[addr + 1] = (uint8_t)(val >> 16);
		ram_data[addr + 2] = (uint8_t)(val >>  8);
		ram_data[addr + 3] = (uint8_t)(val >>  0);
		return;
	}

	fail("invalid write 0x%08x:32 0x%08x", addr, val);
}

void cpu_loop(const char *bios)
{
	hw_init();
	bios_init(bios);

	inf("entering CPU loop");
	m68k_init();
	m68k_set_cpu_type(M68K_CPU_TYPE_68000);
	m68k_pulse_reset();

	uint64_t freq = SDL_GetPerformanceFrequency();
	uint64_t quan = freq / PER_SEC;
	inf("freq %" PRIu64 " quan %" PRIu64, freq, quan);

	bool run = true;

	while (run) {
		uint64_t until = SDL_GetPerformanceCounter() + quan;

		m68k_execute(CPU_FREQ / PER_SEC);
		hw_exec();

		SDL_Event ev;

		while (SDL_PollEvent(&ev) > 0) {
			if (ev.type == SDL_QUIT) {
				run = false;
			}
		}

		while (SDL_GetPerformanceCounter() < until) {
			_mm_pause();
		}
	}

	inf("leaving CPU loop");
}
