/*
 *  Copyright (C) 2017 The Contributors
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or (at
 *  your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful, but
 *  WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 *  General Public License for more details.
 *
 *  A copy of the GNU General Public License can be found in the file
 *  "gpl.txt" in the top directory of this repository.
 */

#include <all.h>

typedef struct {
	const char *name;
	int32_t *flag;
} verb_flag_t;

static verb_flag_t verb_flags[] = {
	{ "sdl", &sdl_verbose },
	{ "cpu", &cpu_verbose },
	{ "fpu", &fpu_verbose },
	{ "vid", &vid_verbose },
	{ "tim", &tim_verbose },
	{ "lcd", &lcd_verbose },
	{ "ser", &ser_verbose },
	{ "mid", &mid_verbose },
	{ "fdd", &fdd_verbose },
	{ "snd", &snd_verbose },
	{ "led", &led_verbose },
	{ "kbd", &kbd_verbose }
};

const char *bios = "bios.abs";
const char *disk = "buchla.disk";

SDL_atomic_t run;

static void usage(FILE *fh)
{
	fprintf(fh, "usage: buchla [-h] [-v comp [-v comp [...]]] [-b bios] [-d disk]\n");
	fprintf(fh, "where comp is one of: ");

	for (int32_t i = 0; i < ARRAY_COUNT(verb_flags); ++i) {
		fprintf(fh, "%s, ", verb_flags[i].name);
	}

	fprintf(fh, "all\n");
}

static void parse_args(int32_t argc, char *argv[])
{
	for (int32_t i = 0; i < argc; ++i) {
		if (strcmp(argv[i], "-h") == 0) {
			usage(stdout);
			exit(0);
		}

		if (strcmp(argv[i], "-b") == 0) {
			if (++i == argc) {
				usage(stderr);
				fprintf(stderr, "missing argument to -b\n");
				exit(1);
			}

			bios = argv[i];
			continue;
		}

		if (strcmp(argv[i], "-d") == 0) {
			if (++i == argc) {
				usage(stderr);
				fprintf(stderr, "missing argument to -d\n");
				exit(1);
			}

			disk = argv[i];
			continue;
		}

		if (strcmp(argv[i], "-v") == 0) {
			if (++i == argc) {
				usage(stderr);
				fprintf(stderr, "missing argument to -v\n");
				exit(1);
			}

			int32_t k;

			if (strcmp(argv[i], "all") == 0) {
				for (k = 0; k < ARRAY_COUNT(verb_flags); ++k) {
					++*verb_flags[k].flag;
				}

				continue;
			}

			for (k = 0; k < ARRAY_COUNT(verb_flags); ++k) {
				if (strcmp(argv[i], verb_flags[k].name) == 0) {
					++*verb_flags[k].flag;
					break;
				}
			}

			if (k == ARRAY_COUNT(verb_flags)) {
				usage(stderr);
				fprintf(stderr, "invalid argument to -v: %s\n", argv[i]);
				exit(1);
			}

			continue;
		}
	}
}

static int32_t cpu_thread(void *data)
{
	(void)data;

	cpu_loop();
	return 0;
}

int32_t main(int32_t argc, char *argv[])
{
	parse_args(argc, argv);
	sdl_init();
	cpu_init();

	SDL_AtomicSet(&run, 1);
	SDL_Thread *thr = SDL_CreateThread(cpu_thread, "cpu", NULL);

	if (thr == NULL) {
		fail("SDL_CreateThread() failed: %s", SDL_GetError());
	}

	sdl_loop();
	SDL_WaitThread(thr, NULL);

	cpu_quit();
	sdl_quit();
	return 0;
}
