/*
   =============================================================================
	scgoto.c -- position display at a given frame time
	Version 48 -- 1988-09-23 -- D.N. Lynx Crowe
   =============================================================================
*/

#define	DEBUG_GO	0	/* define non-zero for sc_goto() debug */
#define	CHECKPTR	0	/* define non-zero to check pointers */

#include "hwdefs.h"
#include "stddefs.h"
#include "graphdef.h"
#include "score.h"
#include "scfns.h"
#include "memory.h"
#include "vsdd.h"
#include "slice.h"
#include "midas.h"

#include "vsddsw.h"

extern void clrnl(void);
extern void clrsctl(void);
extern void dsrpmod(void);
extern void dssect(void);
extern void quiet(int16_t _1, int16_t _2);
extern void sc_adv(void);
extern void scupd(void);
extern void sdwins(void);
extern void se_disp(struct s_entry *ep, int16_t sd, struct gdsel *gdstb[], int16_t cf);
extern void sreset(void);

/* variables defined elsewhere */

#if	DEBUG_GO
extern	short	verbose, testing;
#endif

extern	int16_t	recsw;
extern	int16_t	ndisp;
extern	int16_t	sd;
extern	int16_t	swctrl;
extern	int16_t	swflag;

extern	struct	gdsel	*gdfsep;

extern	struct	gdsel	*gdstbc[];
extern	struct	gdsel	*gdstbn[];

/* 
*/

/*
   =============================================================================
	sc_goto(tval) -- position score display at time 'tval'
   =============================================================================
*/

int16_t sc_goto(int32_t tval)
{
	register struct gdsel	*gdsp;
	register struct s_entry *rp;
	register int32_t	tf, rt;
	register int16_t	mod48 = 48;

#if	CHECKPTR
	Pcheck(p_fwd, "p_fwd - sc_goto entry");
	Pcheck(p_ctr, "p_ctr - sc_goto entry");
	Pcheck(p_bak, "p_bak - sc_goto entry");
	Pcheck(p_cur, "p_cur - sc_goto entry");
#endif

	/* quick check of pointers so we don't crash */

	if ((p_fwd EQ E_NULL) OR (p_cur EQ E_NULL) OR
	    (p_bak EQ E_NULL) OR (p_ctr EQ E_NULL))
		return(FAILURE);

	if (v_regs[5] & 0x0180)		/* setup for VSDD bank 0 */
		vbank(0);

	sd     = D_FWD;			/* set display and scroll direction */
	swctrl = FALSE;			/* stop scroll wheel */
	swflag = FALSE;			/* ... */

	recsw = FALSE;			/* force play mode on goto */
	dsrpmod();			/* update video and LCD displays */

	if (ndisp EQ 2)
		sreset();		/* reset highlighting if score is up */

	quiet(-1, -1);			/* quiet the instrument */
	clrnl();			/* clear note entry lists */
	clrsctl();			/* clear slice control data */

	t_bak = tval - TO_BAK;		/* set target time at p_bak */
	t_fwd = t_bak;			/* set target time at p_fwd */
	t_ctr = tval;			/* set target time at p_ctr */
	t_cur = tval;			/* set target time at p_cur */

	p_bak = frfind(t_bak, 0);	/* position p_bak at t_bak */
	p_cur = frfind(t_cur, 1);	/* position p_cur at t_cur */
	p_fwd = frfind(t_fwd, 1);	/* position p_fwd at t_fwd */

/* 
*/
	/* reset the display pointers to the target time */

	if ((t_fwd LE 0) AND (p_fwd->e_type EQ EV_SCORE))
		p_fwd = p_fwd->e_fwd;	/* skip score header */

	rp = p_fwd;			/* current forward event pointer */
	rt = t_fwd;			/* current forward event time */
	tf = tval + TO_FWD;		/* target event time */

#if	DEBUG_GO
	if (verbose) {

		printf("## sc_goto(%8ld) ENTRY - tf: %8ld\n", tval, tf);

		printf("  t_bak: %8ld  t_ctr: %8ld  t_fwd: %8ld  t_cur: %8ld\n",
			t_bak, t_ctr, t_fwd, t_cur);

		printf("  p_bak: %08lX  p_ctr: %08lX  p_fwd: %08lX  p_cur: %08lX\n",
			p_bak, p_ctr, p_fwd, p_cur);
	}
#endif

/* 
*/

	while (rt++ LT tf) {		/* advance p_fwd chain to tf */

		if (rp->e_type NE EV_FINI) {	/* don't pass end of score */

			while (rp->e_time LE rt) {	/* check event time */

				if (ndisp EQ 2)		/* display event */
					se_disp(rp, D_FWD, gdstbn, 0);

				rp = rp->e_fwd;	/* point at next event */

				if (rp->e_type EQ EV_FINI)	/* done if at end */
					break;
			}
		}

		if (ndisp EQ 2) {

			if (0 EQ (rt % mod48)) {	/* handle beat markers */

				if ((struct gdsel *)NULL NE (gdsp = gdfsep)) {

					gdfsep = gdsp->next;

					gdsp->next = gdstbn[12];
					gdsp->note = 0x1111;
					gdsp->code = 1;

					gdstbn[12] = gdsp;
				}
			}

			sc_adv();	/* scroll the display */
		}
	}

	p_fwd  = rp;		/* update p_fwd for next event */
	t_fwd  = tf;		/* update t_fwd */

/* 
*/
	/* execute & display things at current time to start things out right */

	if (ndisp EQ 2)		/* if score is up ... */
		dssect();	/* display section */

	rp = p_cur;		/* current event pointer */
	rt = t_cur;		/* current event time */

	if ((rt LE 0) AND (rp->e_type EQ EV_SCORE))	/* skip score header */
		rp = rp->e_fwd;

	if (rp->e_type NE EV_FINI) {	/* if we aren't at end of score */

		while (rp->e_time EQ rt) {	/* do events at current time */

			se_exec(rp, D_FWD);		/* execute event */

			if (ndisp EQ 2)
				se_disp(rp, D_FWD, gdstbc, 1);	/* update center slice */

			rp = rp->e_fwd;			/* point at next event */

			if (rp->e_type EQ EV_FINI)	/* done if at end */
				break;
		}
	}

	p_cur = rp;		/* update p_cur */
	p_ctr = rp;		/* update p_ctr */

	if (ndisp EQ 2) {		/* if score is up ... */

		scupd();	/* update event display */
		sdwins();	/* refresh windows */
	}

/* 
*/

#if	DEBUG_GO
	if (verbose) {

		printf("## sc_goto(%8ld) EXIT\n");

		printf("  t_bak: %8ld  t_ctr: %8ld  t_fwd: %8ld  t_cur: %8ld\n",
			t_bak, t_ctr, t_fwd, t_cur);

		printf("  p_bak: %08lx  p_ctr: %08lx  p_fwd: %08lx  p_cur: %08lx\n",
			p_bak, p_ctr, p_fwd, p_cur);
	}
#endif

#if	CHECKPTR
	Pcheck(p_fwd, "p_fwd - sc_goto exiting");
	Pcheck(p_ctr, "p_ctr - sc_goto exiting");
	Pcheck(p_bak, "p_bak - sc_goto exiting");
	Pcheck(p_cur, "p_cur - sc_goto exiting");
#endif

	return(SUCCESS);
}

/* 
*/

/*
   =============================================================================
	sc_refr() -- refresh the display to a particular time
   =============================================================================
*/

int16_t sc_refr(int32_t t)
{
	int16_t oldrs, rc;

	oldrs = recsw;		/* save recsw */
	rc = sc_goto(t);	/* refresh the display via a goto */
	recsw = oldrs;		/* restore recsw */
	dsrpmod();		/* update display of recsw */
	return(rc);		/* return status from sc_goto */
}
